#!/usr/bin/python3
# -*- coding: utf-8 -*-
# pylint: disable=invalid-name,line-too-long
"""
    icdx-monitor.py - view/monitor configured and running services
"""

import os
import sys
import time
import math
import getopt
import smtplib
import logging
import logging.handlers
import requests


# ICDx uses self signed cert by default and we don't really need to be
# concerned with cert validation for the tasks used here, so disable warnings
from urllib3.exceptions import InsecureRequestWarning
requests.packages.urllib3.disable_warnings( category = InsecureRequestWarning )


#################################################
# Customizable Settings
#################################################

# Domain and port used in ICDx access URL
icdxDomain = "localhost"
icdxPort = "443"

# Protocol used, either "http" or "https" ("https" by default)
icdxProtocol = "https"

# Key generated from ICDx for API access
apiKey = ""

# Time span in seconds between polling
# default 8
idleTimer = 8

# Number of attempts to start a service before halting
# 0 will continue to attempt without halting
# default 8
maxAttempts = 8

# Increase the retry timeout by this exponent between attempts
# i.e. retry at retrySpread ^ attemptX
# default 8
retrySpread = 8

# If spread between retries exceeds this, clamp further retries here
# in hours, default 24 (once a day)
maxRetryIdle = 24

# file to store the Watch list
icdxWatchFile = "/tmp/ICDxWatchFile.txt"

# Email information for failure alerts
#  Note: alert mail will not be attempted if mailHost value is empty
mailHost = ''
mailPort = 25
mailFrom = ''
rcptTo = [ '' ]
subject = 'ICDx Service Monitor Alert'

# Logging level
logLevel = logging.INFO



#################################################
# End of Customizable Settings
#################################################



icdxSite = icdxProtocol + "://" + icdxDomain + ":" + icdxPort
maxRetryIdle *= 3600 # mult by seconds per hour


# =====================================
class Services() :
    """ ICDx Services API interaction class

        Params:
            apiKey (str): The API access key obtained from ICDx
    """
# -----------------
# pylint: disable=too-many-instance-attributes
    def __init__( self, key ) :
        self.UUID       = 'uuid'
        self.PATH       = 'path'
        self.NAME       = 'name'
        self.COLLECTOR  = 'collector'
        self.FORWARDER  = 'forwarder'
        self.ADAPTER    = 'adapter'
        self.OTHER      = 'other'
        self.ALL        = 'all'
        self.apiKey     = key
        self.pathIdx = 0
        self.dataIdx = 1
        self.svcRunningParams   = [ '/launcher/manage', '"id":2' ]
        self.svcExistsParams    = [ '/launcher/manage', '"id":1, "path":""' ]
        self.svcStartParams     = [ '/launcher/manage', '"id":10, "path":"{}", "uuid":"{}"' ]
        self.logPostEvent       = [ '/logger/events' ]
        self.requestHeaders = {
            'Authorization' : 'Basic ' + self.apiKey,
            'Content-Type' : 'application/json',
            'Cache-Control' : 'no-cache'
        }
        self.services = [ 'adapter', 'collector', 'forwarder', 'other', 'all' ]


    # =====================================
    def getAPIResponse( self, path, data, *dataVars ) :
        """ Compile request, post request to API, return response

            Params:
                path (str): API URL path to be used for this request
                data (str): data string to be posted to the API
                *dataVars (list): arguments for data string formatting

            Returns:
                API response
        """
    # -----------------
        dataStr = "{"

        if dataVars :
            dataStr += data.format( *dataVars )
        else :
            dataStr += data

        dataStr += "}"

        try:
            response = requests.post( icdxSite + "/api/dx" + path,
                verify = False,
                headers = self.requestHeaders,
                data = dataStr )
        except requests.RequestException as re :
            print("API connection error: " + str( re ) )
            sys.exit()

        return response



    # =====================================
    def getServiceList( self, sType ) :
        """ Retrieve list of configured services

        Params:
            sType (str): Service name: collector, forwarder, all

        Returns:
            List of dicts for found services, [ { self.UUID : uuid, self.PATH : path, self.NAME : name }, ... ]
        """
    # -----------------
        getAll = False
        svcList = []

        if sType not in self.services :
            print( "\nList: Please specify one of: {}\n".format( ' '.join( map( str, self.services ) ) ) )
            return svcList

        if sType == 'all' :
            getAll = True

        response = self.getAPIResponse(
            self.svcExistsParams[ self.pathIdx ],
            self.svcExistsParams[ self.dataIdx ]
        )

        for service in response.json()[ 'list' ] :
            if getAll or service[ 'path' ].split( '/' )[ 0 ] == sType :
                svcList.append( {
                    self.UUID : service[ 'uuid' ],
                    self.PATH : service[ 'path' ],
                    self.NAME : service[ 'name' ]
                } )

        return svcList



    # =====================================
    def getRunningServices( self, sType ) :
        """ Retrieve list of running services

        Params:
            sType (str): type of service to list - all, collector, forwarder, uuid

        Returns:
            List of dicts for found services, [ { self.UUID : uuid, self.PATH : path, self.NAME : name }, ... ]
        """
    # -----------------
        svcList = []

        response = self.getAPIResponse(
            self.svcRunningParams[ self.pathIdx ],
            self.svcRunningParams[ self.dataIdx ]
        )

        for service in response.json()[ 'list' ] :
            # go through the list of services and parse out the ones to display
            found = False
            if sType == "all" :
                found = True
            elif sType == "collector" :
                if service[ 'path' ].split( '/' )[ 0 ] == "collector" :
                    found = True
            elif sType == "forwarder" :
                if service[ 'path' ].split( '/' )[ 0 ] == "forwarder" :
                    found = True
            elif service[ 'uuid' ] == sType :
                found = True

            if found :
                svcList.append( {
                    self.UUID : service[ 'uuid' ],
                    self.PATH : service[ 'path' ],
                    self.NAME : service[ 'name' ]
                } )

        return svcList





# =====================================

# =====================================
def printHelp() :
    """ Print usage information """
# -----------------

    print( """
 
    Usage:
        -l, --list [all|collector|forwarder]

            all: display all services
            collector: display collector services
            forwarder: display forwarder services

        -r, --running [<uuid>|all|collector|forwarder]

            <uuid>: display service info for specific uid, if running
            all: display service info of all running services
            collector: display service info of all running collectors
            forwarder: display service info of all running forwarders

        -w, --watch [<uuid>|list|start]

            <uuid>: toggle the service's inclusion in the watch list
            list: display the watch list
            start: start montioring the services in the watch list; for example, icdx-monitor.py -w start&

        -h, --help

            Print this help information

      """ )



# =====================================
def monitorAlert( logger, msg, *vals ) :
    """ Create an alert message, log to the local syslogger
        and optionally email if global var mailHost is defined

        Params:
            logger (logger object): used for logging results
            msg (str): String format alert message ( i.e. "alert: {}".format(*vals) )
            *vals (list): arguments to use in string format
    """
# -----------------
    messageBody = msg.format( *vals )

    # log the alert
    logger.warn( messageBody )

    # skip email if not configured
    if mailHost != '' :
        # send alert email
        smtpMessage = "From: {}\r\n".format( mailFrom )
        smtpMessage += "To: {}\r\n".format( ", ".join( rcptTo ) )
        smtpMessage += "Subject: {}\r\n".format( subject )
        smtpMessage += "\r\n{}\r\n".format( messageBody )

        try :
            smtpObj = smtplib.SMTP( mailHost, mailPort )
            smtpObj.sendmail( mailFrom, rcptTo, smtpMessage )
            smtpObj.quit()

        except : # pylint: disable=bare-except
            logger.error( "Unable to send email : %s", str( sys.exc_info()[ 1 ] ) )



# =====================================
def doWatch( services, arg ) :
    """ Add/remove or list watched services, or start daemon process

    Params:
        arg (str): action to do, one of:
            <uuid>: to add/remove uuid to/from watched service
            list: list watched services
            start: begin monitoring services
    """
# -----------------

    if arg == "start" :
        watchMonitor( services )
    else :
        serviceWatchList( services, arg )



# =====================================
def serviceWatchList( services, arg ) :
    """ Manage the watch list - add/remove or list

    Params:
        arg (str): list or uuid
    """
# -----------------
    watchList = []

    if os.path.isfile( icdxWatchFile ) :
        with open( icdxWatchFile, 'r' ) as watchFile :
            watchList = watchFile.read().splitlines()

    print()

    if arg == "list" :
        print( "Watch List:" )
        print( *watchList, sep = "\n" )
    else :
        # scan the list for the provided uuid
        if arg in watchList :
            # if found drop
            watchList.remove( arg )
        else :
            # if not found
            # then add
            found = False
            response = services.getAPIResponse(
                services.svcExistsParams[ services.pathIdx ],
                services.svcExistsParams[ services.dataIdx ]
            )

            for service in response.json()[ 'list' ] :
                if service[ 'uuid' ] == arg :
                    found = True

            if found :
                watchList.append( arg )
            else :
                print( "'{}' was not found as an existing service uuid.".format( arg ) )
                print()
                sys.exit()


        # print to screen and save
        print( "Watch List:" )
        with open( icdxWatchFile, 'w' ) as watchFile :
            for uuid in watchList :
                watchFile.write( uuid + '\n' )
                print( uuid )

    print()
    print( "If the list has changed, be sure to kill any running proces (killall) and" )
    print( "  re-run this script to reload watch list." )
    print()



# =====================================
def watchMonitor( services ) :
    """ monitor services

        Params:
            services (Services obj): the services object we will be working with
    """
# -----------------
    # pylint: disable=too-many-branches,too-many-statements
    runningServices = {}
    serviceNames = {}

    logger = logging.getLogger( 'icdxServicWatchLogger' )
    logger.setLevel( logging.INFO )
    handler = logging.handlers.SysLogHandler(
        facility=logging.handlers.SysLogHandler.LOG_DAEMON,
        address = '/dev/log'
    )
    logger.addHandler( handler )
    logFormat = '[%(levelname)s] %(filename)s:%(funcName)s:%(lineno)d \"%(message)s\"'
    handler.setFormatter( logging.Formatter( fmt = logFormat ) )


    # read in current watchlist
    if os.path.isfile( icdxWatchFile ) :
        with open( icdxWatchFile, 'r' ) as watchFile :
            watchList = watchFile.read().splitlines()
    else :
        print()
        print( "There is no watch list yet. Use the \"--watch <uuid>\"argument to add a uuid to be watched." )
        print()
        sys.exit()


    print()
    print( "Monitoring..." )
    logger.info( "Starting ICDxServiceMonitor monitoring process" )

    # map service names to UUIDs
    response = services.getAPIResponse(
        services.svcExistsParams[ services.pathIdx ],
        services.svcExistsParams[ services.dataIdx ]
    )

    for service in response.json()[ 'list' ] :
        if service[ 'name' ] != "default" :
            serviceNames[ service[ 'uuid' ] ] = service[ 'name' ]


    while True:
        currentRunning = []

        # get list of currently running services
        response = services.getAPIResponse(
            services.svcRunningParams[ services.pathIdx ],
            services.svcRunningParams[ services.dataIdx ]
        )

        for service in response.json()[ 'list' ] :
            if service[ 'uuid' ] == 'default' :
                continue

            currentRunning.append( service[ 'uuid' ] )

        logger.debug(" Currently running services: %s", str( currentRunning ) )


        if not bool( runningServices ) :
            # runningServices is empty, populate
            logger.info( "Running Services monitor is empty, populating list..." )

            for uuid in watchList :
                runningServices[ uuid ] = dict()
                runningServices[ uuid ][ 'attemptCounter' ] = 0
                runningServices[ uuid ][ 'failed' ] = False
                runningServices[ uuid ][ 'name' ] = serviceNames[ uuid ]
                runningServices[ uuid ][ 'retrySpan' ] = 0


            time.sleep( idleTimer )
            continue

        # poll through runningServices to update current states
        for uuid in runningServices :
            if runningServices[ uuid ][ 'failed' ] :
                # this service has failed, just skip
                continue

            if runningServices[ uuid ][ 'attemptCounter' ] > 0 :
                # service is in failing state
                timeDelta = time.time() - runningServices[ uuid ][ 'lastAttemptTime' ]

                if runningServices[ uuid ][ 'retrySpan' ] < maxRetryIdle :
                    runningServices[ uuid ][ 'retrySpan' ]  = math.pow(
                        retrySpread,
                        runningServices[ uuid ][ 'attemptCounter' ]
                    )

                if timeDelta < runningServices[ uuid ][ 'retrySpan' ] :
                    # still in idle pause, skip for now
                    logger.debug( "%s : not running, pausing before retry", uuid )
                    continue

            if uuid in currentRunning :
                # service is running, reset things
                runningServices[ uuid ][ 'lastAttemptTime' ] = time.time()
                runningServices[ uuid ][ 'failed' ] = False

                if runningServices[ uuid ][ 'attemptCounter' ] > 0 :
                    # this is a service that has previously failed, send a restart alert
                    monitorAlert(
                        logger,
                        "{} ({}) restarted after {} attempts, resetting counters.",
                        uuid,
                        runningServices[ uuid ][ 'name' ],
                        runningServices[ uuid ][ 'attemptCounter' ]
                    )

                    # reset the counter
                    runningServices[ uuid ][ 'attemptCounter' ] = 0
                    runningServices[ uuid ][ 'retrySpan' ] = 0

            else :
                # service is not running, attempt to start
                response = services.getAPIResponse(
                    services.svcExistsParams[ services.pathIdx ],
                    services.svcExistsParams[ services.dataIdx ]
                )

                for service in response.json()[ 'list' ] :
                    if service[ 'uuid' ] == uuid :
                        path = service[ 'path' ]

                response = services.getAPIResponse(
                    services.svcStartParams[ services.pathIdx ],
                    services.svcStartParams[ services.dataIdx ],
                    path,
                    uuid
                )

                if runningServices[ uuid ][ 'attemptCounter' ] >= maxAttempts > 0 :
                    # we've exceeded max attempts, we've failed
                    runningServices[ uuid ][ 'failed' ] = True
                    monitorAlert(
                        logger,
                        "{} ({}) failed to start after {} attempts, halting further tries",
                        uuid,
                        runningServices[ uuid ][ 'name' ],
                        runningServices[ uuid ][ 'attemptCounter' ]
                    )
                    continue


                # service is failing, update counters
                runningServices[ uuid ][ 'lastAttemptTime' ] = time.time()

                # to keep the exponential retry from ballooning, do not update counter if above max
                #if runningServices[ uuid ][ 'retrySpan' ] <= maxRetryIdle :
                runningServices[ uuid ][ 'attemptCounter' ] += 1

                monitorAlert(
                    logger,
                    "{} ({}) not running, attempting to restart ({}/{})",
                    uuid,
                    runningServices[ uuid ][ 'name' ],
                    runningServices[ uuid ][ 'attemptCounter' ],
                    maxAttempts
                )


        time.sleep( idleTimer )


# ====================================================


def main( argv ):
    """ main(argv):
        read the arguments, run the commands
    """
    services = Services( apiKey )

    try:
        # pylint: disable=unused-variable
        opts, args = getopt.getopt(argv,"hl:r:w:",
            ["help","list=","running=","watch="])
    except getopt.GetoptError:
        printHelp()
        sys.exit(2)

    if len( opts ) == 0 :
        opts = [("-h","")]

    for opt, arg in opts:
        arg = arg.strip()
        if opt in ( '-h', '--help' ):
            printHelp()
            sys.exit()

        elif opt in ( '-l', '--list' ):
            svcList = services.getServiceList( arg )

            print()

            for svc in svcList :
                print( "{:36}  {:40}  {}".format(
                    svc[ services.UUID ],
                    svc[ services.PATH ],
                    svc[ services.NAME ]
                ) )

            print()

        elif opt in ( '-r', '--running' ):
            svcList = services.getRunningServices( arg )

            print()

            for svc in svcList :
                print( "{:36}  {:40}  {}".format(
                    svc[ services.UUID ],
                    svc[ services.PATH ],
                    svc[ services.NAME ]
                ) )

            print()

        elif opt in ( '-w', '--watch' ):
            doWatch( services, arg )

        else :
            printHelp()







if __name__ == "__main__" :
    try :
        main( sys.argv[ 1: ] )
    except KeyboardInterrupt :
        sys.exit( 0 )
